/**
 * php_dhtml
 * Copyright (C) 2004-2005  Eric Colinet <ecolinet - gmail - com>
 * http://wildphp.free.fr/wiki/doku.php?id=php_dhtml:index
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "php_dhtml.h"
#include "dhtml_api.h"
#include "resource.h"

#define le_dhtml_resource_name  "DHTML resource"
static int le_dhtml_resource;

typedef struct _dhtml_resource {
	DWORD th_id;
	HANDLE th_handle;
	HWND hwnd;
} dhtml_resource;


void _php_dhtml_destruction_handler(zend_rsrc_list_entry *rsrc TSRMLS_DC) {
	dhtml_resource *vi= (dhtml_resource *) rsrc->ptr;
	dhtml_post_message(vi->th_id, "quit", NULL, NULL, NULL ); // exit
	efree(vi);
}


/* proto res dhtml_create() 
   Return a resource pointing to a newly created DHTML Window or NULL if something goes wrong
*/
PHP_FUNCTION(dhtml_create)
{
	long thid; 
	dhtml_resource *vi;
	HANDLE thh; HWND hwnd;

	thid= dhtml_create(&thh, &hwnd);
	if( !thid ) RETURN_NULL();
	
	vi= (dhtml_resource*)emalloc(sizeof(dhtml_resource));
	vi->th_id= thid;
	vi->th_handle= thh;
	vi->hwnd= hwnd;
	ZEND_REGISTER_RESOURCE( return_value, vi, le_dhtml_resource );
}



/* proto bool/object dhtml_get_event( [wait=FALSE] ) 
   Return an event (id, type, type, value, extra).
   If no event is pending and wait if false return NULL.
   Return FALSE if something goes wrong.
*/
PHP_FUNCTION(dhtml_get_event)
{
	long wait=1;
	dhtml_event *event;

	if( zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "|l", &wait) == FAILURE )
		RETURN_FALSE

	event= dhtml_get_event(wait);
	if( !event ) RETURN_NULL()

	/* Event */
	object_init(return_value);
	add_property_long(return_value, "id", event->id);
	add_property_string(return_value, "type",  (char*)event->type,  1);
	if( event->value )
		add_property_string(return_value, "value", (char*)event->value, 1);
	else
		add_property_string(return_value, "value", "", 1);
	if( event->extra )
		add_property_string(return_value, "extra", (char*)event->extra, 1);
	else
		add_property_string(return_value, "extra", "", 1);

}

/* proto bool dhtml_set( handle, type[, value, extra] ) 
	Set a property on a window
*/
PHP_FUNCTION(dhtml_set)
{
	zval * handle;
	zval *type, *value=NULL, *extra= NULL;
	int ret;
	dhtml_resource *vi;

	if( zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, 
		"rz|zz", &handle, &type, &value, &extra) == FAILURE )
		RETURN_FALSE

	ZEND_FETCH_RESOURCE(vi, dhtml_resource*, &handle, -1, le_dhtml_resource_name, le_dhtml_resource);

	convert_to_string_ex( &type  );
	if( value )
		convert_to_string_ex( &value );
	if( extra )
		convert_to_string_ex( &extra );

	ret= dhtml_post_message( vi->th_id, Z_STRVAL_P(type), value?Z_STRVAL_P(value):NULL, extra?Z_STRVAL_P(extra):NULL, NULL ); 

	RETURN_BOOL(ret);
}


/* proto bool/string dhtml_get( handle, type[, value] ) 
	Get a property on a window
*/
PHP_FUNCTION(dhtml_get)
{
	zval * handle;
	zval *type, *value=NULL;
	int ret;
	char *retval= NULL;
	dhtml_resource *vi;

	if( zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, 
		"rz|z", &handle, &type, &value) == FAILURE )
		RETURN_FALSE
	
	ZEND_FETCH_RESOURCE(vi, dhtml_resource*, &handle, -1, le_dhtml_resource_name, le_dhtml_resource);
	convert_to_string_ex( &type  );

	ret= dhtml_post_message( vi->th_id, Z_STRVAL_P(type), value?Z_STRVAL_P(value):NULL, NULL, &retval ); 

	if( retval ) {
		ZVAL_STRING(return_value, retval, 1);
		free(retval);
	}
	else
		RETURN_BOOL(ret);
}


/* proto int dhtml_get_id( handle ) 
	Get the thread id of the window
*/
PHP_FUNCTION(dhtml_get_id)
{
	zval * handle;
	dhtml_resource *vi;

	if( zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, 
		"r", &handle) == FAILURE )
		RETURN_FALSE
	
	ZEND_FETCH_RESOURCE(vi, dhtml_resource*, &handle, -1, le_dhtml_resource_name, le_dhtml_resource);

	RETVAL_LONG(vi->th_id);
}


/* proto int dhtml_get_hwnd( handle ) 
	Get the HWND of the window
*/
PHP_FUNCTION(dhtml_get_hwnd)
{
	zval * handle;
	dhtml_resource *vi;

	if( zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, 
		"r", &handle) == FAILURE )
		RETURN_FALSE
	
	ZEND_FETCH_RESOURCE(vi, dhtml_resource*, &handle, -1, le_dhtml_resource_name, le_dhtml_resource);

	RETVAL_LONG((long)vi->hwnd);
}


/* proto object dhtml_get_desktop_size() 
	Return the size of the user desktop in an object (sx, sy)
*/
PHP_FUNCTION(dhtml_get_desktop_size)
{
	long x, y;
	dhtml_get_destop_size(&x, &y);
	object_init(return_value);
	add_property_long(return_value, "sx", x);
	add_property_long(return_value, "sy", y);
}


/* proto bool dhtml_tray_icon( handle, long id, int action, [string icon_file, [string tip='']] )
	Add an icon (icon_file) in the 'Windows Notification Area' also called 'The Tray'.
	$id is the id of the tray icon
	$tip is the ToolTip text
	$icon_file is the icon file
	$action can take 0=add 1=modify -1=delete
*/
PHP_FUNCTION(dhtml_tray_icon)
{
	zval * handle;
	dhtml_resource *vi;
	NOTIFYICONDATA nid;
	long id, tip_len, icon_file_len=0;
	char *tip= "", *icon_file= NULL;
	long action= 0, ret;

    if( zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rll|ss", 
		&handle, &id, &action, &icon_file, &icon_file_len, &tip, &tip_len) == FAILURE )
		RETURN_FALSE;

	ZEND_FETCH_RESOURCE(vi, dhtml_resource*, &handle, -1, le_dhtml_resource_name, le_dhtml_resource);

	// Compatibility - use only pre-5
	nid.cbSize= sizeof(NOTIFYICONDATA);
	nid.hWnd= vi->hwnd;
	nid.uID= id; // User ID
	nid.uCallbackMessage= WM_WIN32STD_TRAY_ICON;
	nid.uFlags= NIF_MESSAGE;
	nid.hIcon= NULL;
	if( tip_len ) 
		nid.uFlags|= NIF_TIP;
	if( icon_file_len )  {
		nid.uFlags|= NIF_ICON;
		nid.hIcon= LoadImage(NULL, icon_file, IMAGE_ICON, 0, 0, LR_DEFAULTSIZE|LR_LOADFROMFILE|LR_LOADTRANSPARENT);
//		LoadImage(NULL, icon_file, IMAGE_ICON, 0, 0, LR_DEFAULTSIZE|LR_LOADTRANSPARENT|LR_SHARED);// Res
		if( !nid.hIcon ) {
			zend_error( E_WARNING, "dhtml_tray_icon : bad icon filename" );
			RETURN_FALSE;
		}
	}
	else if( action==0 ) {
		nid.hIcon= LoadIcon(NULL, IDI_APPLICATION);
	}

	strncpy(nid.szTip, tip, 63); nid.szTip[63]= 0;
	
	if( !action )
		ret= Shell_NotifyIcon(NIM_ADD, &nid);
	else if( action==1 )
		ret= Shell_NotifyIcon(NIM_MODIFY, &nid);
	else
		ret= Shell_NotifyIcon(NIM_DELETE, &nid);
	
	if( !ret ) {
		zend_error( E_WARNING, "dhtml_tray_icon : can't notify tray icon");
		RETURN_FALSE;
	}

	RETURN_BOOL( ret );
}


/* ZEND_DECLARE_MODULE_GLOBALS(dhtml) */

static int le_dhtml;

/* {{{ dhtml_api[]
 */
function_entry dhtml_api[] = {
	ZEND_FE(dhtml_create,	NULL) 
	ZEND_FE(dhtml_get_event,	NULL) 
	ZEND_FE(dhtml_set, NULL) 
	ZEND_FE(dhtml_get, NULL) 
	ZEND_FE(dhtml_get_id, NULL) 
	ZEND_FE(dhtml_get_hwnd, NULL) 
	ZEND_FE(dhtml_get_desktop_size, NULL) 
	ZEND_FE(dhtml_tray_icon, NULL) 

	{NULL, NULL, NULL}
};
/* }}} */

#if ZEND_MODULE_API_NO < 20010901
# error Module made for PHP 4.1.x
#endif

/* {{{ dhtml_module_entry
 */
zend_module_entry dhtml_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	"dhtml",
	dhtml_api,
	ZEND_MINIT(dhtml),
	ZEND_MSHUTDOWN(dhtml),
	NULL,
	NULL,
#if ZEND_MODULE_API_NO >= 20010901
	ZEND_MINFO(dhtml),
#endif
    "0.2", /* Replace with version number for your extension */
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_DHTML
ZEND_GET_MODULE(dhtml)
#endif

/* {{{ PHP_INI
 * /
PHP_INI_BEGIN()
    STD_PHP_INI_ENTRY("dhtml.value",      "42", PHP_INI_ALL, OnUpdateInt, global_value, zend_dhtml_globals, dhtml_globals)
    STD_PHP_INI_ENTRY("dhtml.string", "foobar", PHP_INI_ALL, OnUpdateString, global_string, zend_dhtml_globals, dhtml_globals)
PHP_INI_END()
/* }}} */

/* {{{ php_dhtml_init_globals
 * /
static void php_dhtml_init_globals(zend_dhtml_globals *dhtml_globals)
{
	dhtml_globals->value = 0;
	dhtml_globals->string = NULL;
}
/* }}} */

/* {{{ PHP_MINIT_FUNCTION
 */	
ZEND_MINIT_FUNCTION(dhtml)
{
	/*
	ZEND_INIT_MODULE_GLOBALS(dhtml, php_dhtml_init_globals, NULL);
	REGISTER_INI_ENTRIES();
	*/
	le_dhtml_resource= zend_register_list_destructors_ex(_php_dhtml_destruction_handler, NULL, le_dhtml_resource_name, module_number);

	REGISTER_LONG_CONSTANT("DHTML_TRAY_ADD", 0, CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("DHTML_TRAY_UPDATE", 1, CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("DHTML_TRAY_REMOVE", -1, CONST_CS | CONST_PERSISTENT);

	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MSHUTDOWN_FUNCTION
 */
ZEND_MSHUTDOWN_FUNCTION(dhtml)
{
	/*	UNREGISTER_INI_ENTRIES(); */
	return SUCCESS;
}
/* }}} */


/* {{{ PHP_MINFO_FUNCTION
 */
ZEND_MINFO_FUNCTION(dhtml)
{
	php_info_print_table_start();
	php_info_print_table_header(1, "PHP_DHTML extension");
	php_info_print_table_row(1, "http://wildphp.free.fr/wiki/doku.php?id=php_dhtml:index");
#if PHP_DHTML_MFC_APP_EMBEDED
	php_info_print_table_row(2, "MFC App embeded", "Yes");
#else
	php_info_print_table_row(2, "MFC App embeded", "Not in php_dhtml.dll");
#endif
	php_info_print_table_end();

	/*
	DISPLAY_INI_ENTRIES();
	*/
}
/* }}} */
